<?php
require_once 'config.php';

// 获取最新的批次号
function getLatestBatchNumber($conn, $moduleCode) {
    $stmt = $conn->prepare("SELECT MAX(batch_number) as max_batch FROM batch_records WHERE module_code = ?");
    $stmt->bind_param("s", $moduleCode);
    $stmt->execute();
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    return $row['max_batch'] ? $row['max_batch'] + 1 : 1;
}

// 产品型号数据
$moduleData = [
    'MGE1' => ['name' => 'LR1302 LoRaWAN Gateway Module（EU868）', 'image' => 'images/MGE1.png'],
    'MGU1' => ['name' => 'LR1302 LoRaWAN Gateway Module（US915）', 'image' => 'images/MGU1.png'],
    'MTR1' => ['name' => 'nRFLR1110 LoRa Transceiver Module', 'image' => 'images/MTR1.png'],
    'MTR2' => ['name' => 'nRFLR1262 LoRa Transceiver Module', 'image' => 'images/MTR2.png'],
    'MTR3' => ['name' => 'nRFLR1121 LoRa Transceiver Module', 'image' => 'images/MTR3.png'],
    'MTR4' => ['name' => 'nRFLRCC68 LoRa Transceiver Module', 'image' => 'images/MTR4.png'],
    'MTR5' => ['name' => 'LR1262 LoRa Transceiver Module', 'image' => 'images/MTR5.png'],
    'MTR6' => ['name' => 'LRCC68 LoRa Transceiver Module', 'image' => 'images/MTR6.png'],
    'MTR7' => ['name' => 'LR1262 LoRaWAN Node Module', 'image' => 'images/MTR7.png'],
    'G0EU1' => ['name' => 'ThinkNode G1（EU868）', 'image' => 'images/G0EU1.png'],
    'G0EU2' => ['name' => 'ThinkNode G3（EU868）', 'image' => 'images/G0EU2.png'],
    'G0US1' => ['name' => 'ThinkNode G1（US915）', 'image' => 'images/G0US1.png'],
    'G0US2' => ['name' => 'ThinkNode G3（US915）', 'image' => 'images/G0US2.png'],
    'G0AU1' => ['name' => 'ThinkNode G3（AU915）', 'image' => 'images/G0AU1.png'],
    'G0AS1' => ['name' => '非4G网关（AS923）', 'image' => 'images/G0AS1.png'],
    'G4EU1' => ['name' => 'ThinkNode G3（EU868）', 'image' => 'images/G4EU1.png'],
    'G4EU2' => ['name' => 'ThinkNode G1（EU868）', 'image' => 'images/G4EU2.png'],
    'G4US1' => ['name' => 'ThinkNode G3（US915）', 'image' => 'images/G4US1.png'],
    'G4US2' => ['name' => 'ThinkNode G1（US915）', 'image' => 'images/G4US2.png'],
    'G4AU1' => ['name' => 'ThinkNode G3（AU915）', 'image' => 'images/G4AU1.png'],
    'G4AS1' => ['name' => '4G网关（AS923）', 'image' => 'images/G4AS1.png'],
    'TR1' => ['name' => 'ThinkNode M1', 'image' => 'images/TR1.png'],
    'TR2' => ['name' => 'ThinkNode M2', 'image' => 'images/TR2.png'],
    'CPI1' => ['name' => 'CrowPi', 'image' => 'images/CPI1.png'],
    'CPI2' => ['name' => 'CrowPi2', 'image' => 'images/CPI2.png'],
    'CPI3' => ['name' => 'CrowPi3', 'image' => 'images/CPI3.png'],
    'CPIL' => ['name' => 'CrowPi L', 'image' => 'images/CPIL.png'],
    'CTL1' => ['name' => 'Crowtail STEAM Educational Kit for BBC Micro:bit', 'image' => 'images/CTL1.png'],
    'CTL2' => ['name' => 'Crowtail-Starter Kit for Micro:bit', 'image' => 'images/CTL2.png'],
    'CVW1' => ['name' => 'CrowView', 'image' => 'images/CVW1.png'],
    'CVW2' => ['name' => 'CrowView Note', 'image' => 'images/CVW2.png'],
    'CVW3' => ['name' => 'CrowView Yoga', 'image' => 'images/CVW3.png'],
    'CPLB1' => ['name' => 'CrowPanel 7.0\"', 'image' => 'images/CPLB1.png'],
    'CPLB2' => ['name' => 'CrowPanel 5.0\"', 'image' => 'images/CPLB2.png'],
    'CPLA1' => ['name' => 'CrowPanel Advance 7\" HMI', 'image' => 'images/CPLA1.png'],
    'CPLA2' => ['name' => 'CrowPanel Advance 5\" HMI', 'image' => 'images/CPLA2.png']
];
?>
<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>产品SN编码生成系统</title>
    <link href="./web/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css" rel="stylesheet">
    <style>
        .product-image {
            max-width: 200px;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <div class="container mt-5">
        <h2 class="mb-4">产品SN编码生成系统</h2>
        <form id="snForm" method="post" action="generate.php">
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">产品型号</label>
                        <select class="form-select" name="module_code" id="moduleSelect" required>
                            <option value="">请选择模块型号</option>
                                <optgroup label="模块类：网关模块（EU868）">
                                    <option value="MGE1">LR1302 LoRaWAN Gateway Module（EU868）：MGE1</option>
                                </optgroup>
                                <optgroup label="模块类：网关模块（US915）">
                                    <option value="MGU1">LR1302 LoRaWAN Gateway Module（US915）：MGU1</option>
                                </optgroup>
                                <optgroup label="模块类：收发器模块">
                                    <option value="MTR1">nRFLR1110 LoRa Transceiver Module：MTR1</option>
                                    <option value="MTR2">nRFLR1262 LoRa Transceiver Module：MTR2</option>
                                    <option value="MTR3">nRFLR1121 LoRa Transceiver Module：MTR3</option>
                                    <option value="MTR4">nRFLRCC68 LoRa Transceiver Module：MTR4</option>
                                    <option value="MTR5">LR1262 LoRa Transceiver Module：MTR5</option>
                                    <option value="MTR6">LRCC68 LoRa Transceiver Module：MTR6</option>
                                    <option value="MTR7">LR1262 LoRaWAN Node Module：MTR7</option>
                                </optgroup>
                                <optgroup label="网关：非4G网关（EU868）">
                                    <option value="G0EU1">ThinkNode G1（EU868）：G0EU1</option>
                                    <option value="G0EU2">ThinkNode G3（EU868）：G0EU2</option>
                                </optgroup>
                                <optgroup label="网关：非4G网关（US915）">
                                    <option value="G0US1">ThinkNode G1（US915）：G0US1</option>
                                    <option value="G0US2">ThinkNode G3（US915）：G0US2</option>
                                </optgroup>
                                <optgroup label="网关：非4G网关（AU915）">
                                    <option value="G0AU1">ThinkNode G3（AU915）：G0AU1</option>
                                </optgroup>
                                <optgroup label="网关：非4G网关（AS923）">
                                    <option value="G0AS1">非4G网关（AS923）：G0AS1</option>
                                </optgroup>
                                <optgroup label="网关：4G网关（EU868）">
                                    <option value="G4EU1">ThinkNode G3（EU868）：G4EU1</option>
                                    <option value="G4EU2">ThinkNode G1（EU868）：G4EU2</option>
                                </optgroup>
                                <optgroup label="网关：4G网关（US915）">
                                    <option value="G4US1">ThinkNode G3（US915）：G4US1</option>
                                    <option value="G4US2">ThinkNode G3（US915）：G4US2</option>
                                </optgroup>
                                <optgroup label="网关：4G网关（AU915）">
                                    <option value="G4AU1">ThinkNode G3（AU915）：G4AU1</option>
                                </optgroup>
                                <optgroup label="网关：4G网关（AS923）">
                                    <option value="G4AS1">4G网关（AS923）：G4AS1</option>
                                </optgroup>
                            <optgroup label="收发器：频段EU868">
                                <option value="TREU1">ThinkNode M1（EU868）：TREU1</option>
                                <option value="TREU2">ThinkNode M5（EU868）：TREU2</option>
                            </optgroup>
                            <optgroup label="收发器：频段US915">
                                <option value="TRUS1">ThinkNode M1（US915）：TRUS1</option>
                                <option value="TRUS2">ThinkNode M5（US915）：TRUS2</option>
                            </optgroup>
                            <optgroup label="收发器：多频段">
                                <option value="TRAL1">ThinkNode M2: TRAL1</option>
                                <option value="TRAL2">ThinkNode M3（Meshtastic）: TRAL2</option>
                                <option value="TRAL3">ThinkNode M3（Tracker）: TRAL3</option>
                            </optgroup>
                            <optgroup label="CrowPi系列">
                                <option value="CPI1">CrowPi：CPI1</option>
                                <option value="CPI2">CrowPi2：CPI2</option>
                                <option value="CPI3">CrowPi3：CPI3</option>
                                <option value="CPIL">CrowPi L：CPIL</option>
                            </optgroup>
                            <optgroup label="Crowtail系列">
                                <option value="CTL1">Crowtail STEAM Educational Kit for BBC Micro:bit：CTL1</option>
                                <option value="CTL2">Crowtail-Starter Kit for Micro:bit：CTL2</option>
                            </optgroup>
                            <optgroup label="CrowView系列">
                                <option value="CVW1">CrowView：CVW1</option>
                                <option value="CVW2">CrowView Note：CVW2</option>
                                <option value="CVW3">CrowView Yoga：CVW3</option>
                            </optgroup>
                                <optgroup label="CrowPanel系列：Basic类">
                                    <option value="CPLB1">CrowPanel 7.0&quot;：CPLB1</option>
                                    <option value="CPLB2">CrowPanel 5.0&quot;：CPLB2</option>
                                </optgroup>
                                <optgroup label="CrowPanel系列：Advance类">
                                    <option value="CPLA1">CrowPanel Advance 7&quot; HMI：CPLA1</option>
                                    <option value="CPLA2">CrowPanel Advance 5&quot; HMI：CPLA2</option>
                                </optgroup>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">产品图片</label>
                        <div id="productImage"></div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">生产日期</label>
                        <input type="date" class="form-control" name="batch_date" id="batchDate" required>
                        <div id="yearWeek" class="mt-2"></div>
                    </div>
                </div>

                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">批次编码(自动生成)</label>
                        <input type="text" class="form-control" id="batchWeekCode" name="batch_week_code" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">批次编号(自动生成)</label>
                        <input type="text" class="form-control" id="batchNumber" name="batch_number" readonly>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">流水号起始位置</label>
                        <input type="number" class="form-control" name="start_sequence" min="1" max="99999" value="1" required id="startSequenceInput">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">数量</label>
                        <input type="number" class="form-control" name="quantity" min="1" max="99999" required id="quantityInput">
						<div id="sequenceRangeDisplay" class="mt-2"></div>
                    </div>
                </div>
            </div>

            <div class="row mt-3">
                <div class="col-12">
                    <button type="submit" class="btn btn-primary">生成SN编码并导出CSV</button>
                    <button type="button" class="btn btn-info ms-2" id="viewHistoryBtn">查看导出历史记录</button>
					<button type="button" class="btn btn-info ms-2" data-bs-toggle="modal" data-bs-target="#guideModal" title="操作说明">?</button>
                </div>
            </div>

            <div class="row mt-4">
                <div class="col-12">
                    <h3>导出历史记录</h3>
                    <div id="exportHistoryList" class="list-group" style="height:400px; margin-bottom:50px;overflow:auto;">
                        <!-- 历史记录将在这里加载 -->
                    </div>
                    <div id="historyMessage" class="mt-2"></div>
                </div>
            </div>
        </form>
    </div>
	 <!-- 功能指南模态框 -->
    <div class="modal fade" id="guideModal" tabindex="-1" aria-labelledby="guideModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="guideModalLabel">功能指南</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p><strong>产品型号:</strong> 下拉菜单选择产品产品型号</p>
                    <p><strong>年份和周数(4位):</strong> 选择日期，自动填入年份和周数</p>
                    <p><strong>批次(3位):</strong> 选择模块型号后，自动查询上次导出时该模块型号的批次，自动填入本次批次</p>
                    <p><strong>流水号起始位置:</strong> 输入流水号的起始数字</p>
                    <p><strong>数量:</strong> 填入生产数量</p>
                    <p><strong>按键功能:</strong></p>
                    <ul>
                        <li><strong>导出:</strong> 输入完以上需求后点击导出，生成并导出所需的SN码表</li>
                        <li><strong>导出历史:</strong> 每次导出操作后，会自动备份一份excel文件在云服务器。点击导出历史，可查看以前导出的excel文件</li>
                        <li><strong>?:</strong> 查看功能指南，点击X即可关闭</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    <script src="./web/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <script src="./web/zh.js"></script>
    <script>
        const moduleData = <?php echo json_encode($moduleData); ?>;

        document.addEventListener('DOMContentLoaded', function() {
            // 初始化日期选择器
            flatpickr("#batchDate", {
                locale: "zh",
                dateFormat: "Y-m-d",
                onChange: function(selectedDates) {
                    updateYearWeek(selectedDates[0]);
                    updateBatchCode(selectedDates[0]); // 新增：更新批次编码
                }
            });

            // 模块选择变化时更新批次编号
            document.getElementById('moduleSelect').addEventListener('change', function() {
                updateBatchNumber(this.value);
                updateProductImage(this.value);
            });
			// 流水号起始位置和数量变化时更新流水号范围
            document.getElementById('startSequenceInput').addEventListener('input', updateSequenceRange);
            document.getElementById('quantityInput').addEventListener('input', updateSequenceRange);

            // 初始加载时更新流水号范围
            updateSequenceRange();
			
			// 查看导出历史记录按钮点击事件
            document.getElementById('viewHistoryBtn').addEventListener('click', function() {
                loadExportHistory();
            });
        });

        function updateYearWeek(date) {
            const year = date.getFullYear().toString().slice(-2);
            const week = getWeekNumber(date);
            document.getElementById('yearWeek').textContent = `年份和周数：${year}${week}`;
        }
		
        function updateBatchCode(date) {
            const year = date.getFullYear().toString().slice(-2);
            const week = getWeekNumber(date);
            document.getElementById('batchWeekCode').value = `${year}${week}`;
        }

        function getWeekNumber(date) {
            const firstDayOfYear = new Date(date.getFullYear(), 0, 1);
            const pastDaysOfYear = (date - firstDayOfYear) / 86400000;
            return Math.ceil((pastDaysOfYear + firstDayOfYear.getDay() + 1) / 7);
        }
		
		function updateSequenceRange() {
            const startSequence = parseInt(document.getElementById('startSequenceInput').value);
            const quantity = parseInt(document.getElementById('quantityInput').value);
            const displayElement = document.getElementById('sequenceRangeDisplay');

            if (isNaN(startSequence) || isNaN(quantity) || quantity <= 0) {
                displayElement.textContent = '';
                return;
            }

            const endSequence = startSequence + quantity - 1;

            const formattedStart = startSequence.toString().padStart(5, '0');
            const formattedEnd = endSequence.toString().padStart(5, '0');

            displayElement.textContent = `流水号范围：${formattedStart}-${formattedEnd}`;
        }

		async function loadExportHistory() {
            const moduleCode = document.getElementById('moduleSelect').value;
            const historyListDiv = document.getElementById('exportHistoryList');
            const historyMessageDiv = document.getElementById('historyMessage');
            historyListDiv.innerHTML = ''; // 清空现有历史记录
            historyMessageDiv.textContent = ''; // 清空消息

            if (!moduleCode) {
                historyMessageDiv.textContent = '请先选择一个模块型号以便查看历史记录。';
                return;
            }

            try {
                const response = await fetch(`get_export_history.php?module_code=${moduleCode}`);
                const history = await response.json();

                if (history.length > 0) {
                    history.forEach(record => {
                        const item = document.createElement('a');
                        item.href = `#`; // 暂时设置为#，后续点击下载
                        item.classList.add('list-group-item', 'list-group-item-action');
                        item.innerHTML = `
                            <strong>文件名:</strong> ${record.file_name} <br>
                            <strong>日期:</strong> ${record.created_at} <br>
                            <strong>批次编码:</strong> ${record.batch_week_code} <br>
                            <strong>批次编号:</strong> ${String(record.batch_number).padStart(3, '0')} <br>
                            <strong>流水号范围:</strong> ${String(record.start_sequence).padStart(5, '0')}-${String(record.end_sequence).padStart(5, '0')}
                        `;
                        item.addEventListener('click', (e) => {
                            e.preventDefault();
                            // 实现点击下载逻辑
                            window.location.href = `download_history.php?file=${record.file_name}`;
                        });
                        historyListDiv.appendChild(item);
                    });
                } else {
                    historyMessageDiv.textContent = `模块型号 ${moduleCode} 暂无导出历史记录。`;
                }
            } catch (error) {
                console.error('获取导出历史失败:', error);
                historyMessageDiv.textContent = '加载历史记录时发生错误。';
            }
        }
		
        function updateBatchNumber(moduleCode) {
            if (!moduleCode) {
                document.getElementById('batchNumber').value = '';
                return;
            }
            
            fetch(`get_batch.php?module_code=${moduleCode}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('batchNumber').value = String(data.batch_number).padStart(3, '0');
                })
                .catch(error => {
                    console.error('获取批次编号失败:', error);
                    document.getElementById('batchNumber').value = '001'; // 默认值
                });
        }

        function updateProductImage(moduleCode) {
            const imageDiv = document.getElementById('productImage');
            imageDiv.innerHTML = ''; // 清空原有图片
            if (moduleCode && moduleData[moduleCode] && moduleData[moduleCode].image) {
                const img = document.createElement('img');
                img.src = moduleData[moduleCode].image;
                img.alt = moduleData[moduleCode].name;
                img.classList.add('product-image');
                imageDiv.appendChild(img);
            }
        }
    </script>
</body>
</html>